/*  Sucht die Belegungszeiten einer vorgegeben Ressource für ein vorgegebenes Zeitenster raus.
    Also die Zeiten an der eine Ressource beschäftigt bzw. belegt ist.

    Dabei wird beachtet, dass sich mehrere Belegungszeiten überlagern können.
    Die Belegungszeiten werden dabei in die größt möglichen Zeitslots zerlegt, in denen sich nichts an dem Zustand der Ressource
    (z.B. ein Grund für die Belegung kommt hinzu oder fällt weg) ändert.

    Mit dem Parameter "_target_load" können Belegungszeiten ignoriert werden, welche genug Freiraum für eine Abarbeitung des AG bieten.
    Damit das funktioniert, muss die notwendige Auslastung des AG kleiner 1 sein.
    Bei einem vorgegebenen "_target_load" von 0 oder "_checkBlockedTimes" gleich "false" werden nur "Ausschalt"-Zeiten der Ressource zurückgegeben.

    Diese Zeitslots werden mit Daten aus den vorhergehenden und nachfolgenden Zeitslots angereichert.
*/
SELECT tsystem.function__drop_by_regex( 'resource_timeline__select', 'scheduling', _commit => true );
CREATE OR REPLACE FUNCTION scheduling.resource_timeline__select(
      _resource_id        integer,
      _target_load        numeric = 1,
      _scenario           varchar = null,
      _timeframe_start    timestamp = date_trunc( 'day', currenttime() ),
      _timeframe_end      timestamp = date_trunc( 'day', currenttime() ) + interval '1 year',
      _blocktime_cursor   refcursor = null,
      _checkBlockedTimes  bool = true,          -- Bei 'false' DLZ-Terminierung: Ignoriert Blockierungen auf der Ressource durch andere Task*-Einträge. Es werden nur Off-Times beachtet.
      _loglevel           int DEFAULT TSystem.Log_Get_LogLevel( _user => 'yes' )
  ) RETURNS TABLE(
      ids        integer[],
      load       numeric,
      date_start timestamp,
      date_end   timestamp,
      types      scheduling.resource_timeline_blocktype[],
      next_start timestamp,
      next_end   timestamp,
      prev_start timestamp,
      prev_end   timestamp,
      prev_types scheduling.resource_timeline_blocktype[],
      next_types scheduling.resource_timeline_blocktype[],
      prev_load  numeric,
      next_load  numeric
  ) AS $$
  DECLARE

      _r          record;

  BEGIN
      IF _loglevel >= 6 THEN
          RAISE NOTICE '%', format(
                                      'call: scheduling.resource_timeline__select( _resource_id => %L, _target_load => %L, _scenario => %L, _timeframe_start => %L, _timeframe_end => %L, _blocktime_cursor => %L, _checkBlockedTimes => %L, _loglevel => %L )',
                                                                                  _resource_id       , round( _target_load, 4 ), _scenario, _timeframe_start      , _timeframe_end      , _blocktime_cursor      , _checkBlockedTimes      , _loglevel
                            )
          ;

          IF NOT _checkBlockedTimes THEN
              RAISE NOTICE 'DLZ-Terminierung.';
          END IF;
      END IF;

      IF ( _timeframe_end < _timeframe_start ) THEN
          RAISE EXCEPTION USING
              ERRCODE = 'TFAIL',
              MESSAGE = '_timeframe_end < _timeframe_start : "' || _timeframe_end::varchar || '<' || _timeframe_start::varchar || '"',
              HINT = 'Termination not possible within given timeframe';
      END IF;

      -- Lese übergebene Belegungszeiten von Kostenstellen aus externen Quellen über den Refcursor "_blocktime_cursor" ein.
      -- Dies gehört nicht zur öffentlichen API.

      -- DROP TABLE IF EXISTS _tmp_blocktimes;

      SET client_min_messages = warning; -- hide notice already exists

      CREATE TEMP TABLE IF NOT EXISTS _tmp_blocktimes AS
          SELECT ti_id, ti_resource_id, ti_a2_id, ti_date_start, ti_date_end, ti_type, ti_ta_kf, ti_usage
          FROM scheduling.resource_timeline
          LIMIT 0;
      CREATE TEMP TABLE IF NOT EXISTS _tmp_timeline AS SELECT * FROM _tmp_blocktimes LIMIT 0;
      CREATE INDEX IF NOT EXISTS _tmp_timeline__ti_date_range__btree ON _tmp_timeline ( ti_date_start, ti_date_end DESC );

      SET client_min_messages = notice;

      TRUNCATE _tmp_timeline;

      -- Lese zeilenweise die Daten vom Refcursor "_blocktime_cursor" und schreibe diese in die temporäre Tabelle "_tmp_blocktimes".
      IF ( _blocktime_cursor IS NOT NULL ) THEN

          raise notice 'temp cursor provided; splicing data together';

          LOOP

              FETCH _blocktime_cursor INTO _r;

              IF ( NOT FOUND ) THEN
                  EXIT;
              END IF;

              INSERT INTO _tmp_blocktimes
                VALUES (
                  _r.ti_id,
                  _r.ti_a2_id,
                  _r.ti_resource_id,
                  _r.ti_date_start,
                  _r.ti_date_end,
                  _r.ti_type,
                  _r.ti_ta_kf,
                  _r.ti_usage
                );

          END LOOP;
      END IF;

      -- DROP TABLE IF EXISTS _tmp_timeline;
      -- CREATE TEMP TABLE _tmp_timeline AS
      WITH
        -- Belegungszeiten aus externer Quelle (vom Refcursor "_blocktime_cursor") gefiltert auf eine kornkrete Kostenstelle (_resource_id)
        -- und begrenzt auf das Zeitfenster (_timeframe_start, _timeframe_end).
        _blocktimes AS (
            SELECT b.*
              FROM _tmp_blocktimes b
            WHERE
                  b.ti_resource_id = _resource_id
--              AND    tsrange( _timeframe_start , _timeframe_end, '[]' )
--                  && tsrange( b.ti_date_start, b.ti_date_end, '[]' )
                AND _timeframe_start <= b.ti_date_end
                AND _timeframe_end   >= b.ti_date_start
        ),

        -- Belegungszeiten aus der Tabelle "scheduling.resource_timeline" gefiltert auf eine kornkrete Kostenstelle (_resource_id)
        -- und begrenzt auf das Zeitfenster (_timeframe_start, _timeframe_end).
        _raw_data AS (
          SELECT
                tr1.ti_id, tr1.ti_a2_id, tr1.ti_resource_id,
                tr1.ti_date_start, tr1.ti_date_end, tr1.ti_type, tr1.ti_ta_kf, tr1.ti_usage
          FROM scheduling.resource_timeline tr1
          WHERE
                tr1.ti_resource_id = _resource_id
                -- timeframe lookup
                -- TODO AXS analog unten mit <> um Indexe zu nutzen!
--            AND    tsrange( _timeframe_start , _timeframe_end, '[]' )
--                && tsrange( tr1.ti_date_start, tr1.ti_date_end, '[]' )
              AND _timeframe_start <= tr1.ti_date_end
              AND _timeframe_end   >= tr1.ti_date_start
        )

      -- Kombiniere die Belegungszeiten von beiden Quellen (Tabelle "scheduling.resource_timeline" und Refcursor "_blocktime_cursor").
      -- Ggf. filtere auf Zeiten, in denen die Kostenstelle ausser Betrieb ist (bei Parameter "_target_load" = 0).

      INSERT INTO _tmp_timeline
      SELECT
        t.*
      FROM (
        SELECT * FROM _blocktimes
        UNION
        SELECT * FROM _raw_data
      ) t
      WHERE
            -- Bei einer Belastung des AG (_target_load) von 0 oder die DLZ-Terminierung (_checkBlockedTimes=false) werden nur die Einträge der Ausschaltzeiten zurückgegeben.
            -- Bei DLZ-Terminierung wird hier mit Absicht nicht das _target_load auf 0 gesetzt, da das _target_load später als Belastung des AG zurückgegeben wird und dort den urspünglichen Wert liefern soll.
                  ( ti_type IN ( 'off', 'off.day', 'off.time', 'task.blocktime', 'task.buffer' ) )
               OR ( _target_load > 0 AND _checkBlockedTimes );

      --CREATE INDEX _tmp_timeline__a2_id__date_start ON _tmp_timeline ( ti_a2_id, ti_date_start );
      --CREATE INDEX _tmp_timeline__ti_resource_id__ti_date_range__btree ON _tmp_timeline ( ti_resource_id, ti_date_start, ti_date_end DESC );
      --CREATE INDEX _tmp_timeline__ti_date_range__btree ON _tmp_timeline ( ti_date_start, ti_date_end DESC );

      RETURN QUERY
      WITH
        -- Gebe eine Liste von Zeitpunkten zurück, an denen sich am Zustand der Kostentelle etwas ändert.
        -- (Z.B. die Kostenstelle wird belegt oder noch mehr belegt oder wieder frei usw.)
        _times AS (
            SELECT
              DISTINCT _point_in_time
            FROM (
                SELECT ti_date_start AS _point_in_time FROM _tmp_timeline
              UNION
                SELECT ti_date_end AS _point_in_time FROM _tmp_timeline
              UNION
                -- injecting current searchtime
                SELECT _timeframe_start
              UNION
                -- injecting current searchtime
                SELECT _timeframe_end
            ) t
            ORDER BY _point_in_time ASC
        ),

        -- Die Liste der Zeitpunkte wird so zu Zeitintervallen zusammengesetzt, dass die Dauer des vorgegebenen Zeitfensters ausgefüllt ist.
        -- Dabei wird jeweils der aktuelle Zeitpunkt mit dem nächst möglichem Zeitpunkt zu einem Zeitintervall verbunden.
        _intervals AS (
          SELECT
            _point_in_time AS p_start,
            lead( _point_in_time ) over () AS p_end
          FROM _times
        ),

        -- Die Liste der Zeitintervalle wird nun mit der Liste der Belegungszeiten aus den Quellen Tabelle "scheduling.resource_timeline" und Refcursor "_blocktime_cursor" verbunden.
        _data_over_time AS (
          SELECT
            array_agg( c1.ti_id ) AS ti_ids,
            array_agg( c1.ti_type ) AS ti_types,
            coalesce(
              ( SELECT sum( c2.ti_usage )
                FROM _tmp_timeline c2
                WHERE
                      c2.ti_date_start < i.p_end
                  AND c2.ti_date_end > i.p_start
                  -- Variante mit tsrange.
                  -- Ist langsamer als der einfache Vergleich. Indexe über die ti_date_*-Spalten werden aktuell vom
                  -- Algorithmis nicht verwendet und bringen daher keine Besserung.
                  --    tsrange( c2.ti_date_start, c2.ti_date_end, '[]' )
                  --@>  tsrange( i.p_start, i.p_end, '[]' )
              )
              ,0
            ) AS load,
            i.p_start AS ti_date_start,
            i.p_end AS ti_date_end
          FROM _intervals AS i
          JOIN _tmp_timeline AS c1 ON
                                c1.ti_date_start <= i.p_end
                            AND c1.ti_date_end >= i.p_start
                            -- Variante mit tsrange.
                            -- Ist langsamer als der einfache Vergleich. Indexe über die ti_date_*-Spalten werden aktuell vom
                            -- Algorithmis nicht verwendet und bringen daher keine Besserung.
                            --    tsrange( c1.ti_date_start, c1.ti_date_end, '[]' )
                            --@>  tsrange( i.p_start, i.p_end, '[]' )
          GROUP BY i.p_start, i.p_end
        ),

        -- Es werden die Grenzen des vorgegeben Zeitfensters als Zeitintervalle mit der Länge 0 davor- bzw. dahintergehängt.
        _data_over_time_with_border as (
            -- obere Grenze des vorgegebenen Zeitfensters
            SELECT
              NULL AS ti_ids,
              array[ 'off' ]::scheduling.resource_timeline_blocktype[] AS ti_types,
              1 AS load,
              _timeframe_start AS ti_date_start,
              _timeframe_start AS ti_date_end
            --
            UNION ALL
            -- Daten
            SELECT * FROM _data_over_time
            --
            UNION ALL
            -- untere Grenze des vorgegebenen Zeitfensters
            SELECT
              null as ti_ids,
              array[ 'off' ]::scheduling.resource_timeline_blocktype[] AS ti_types,
              1 AS load,
              _timeframe_end AS ti_date_start,
              _timeframe_end AS ti_date_end
        ),

        -- Die nachfolgende und die vorhergehende Belegungszeit errechnen. Notwendig, um die Größe der freien Zeitfenster berechnen zu können.
        prevnext AS (
            SELECT
              *,
              -- nachfolgende Belegungszeit
              lead( d.ti_date_start ) OVER( ORDER BY d.ti_date_start, d.ti_date_end ) AS next_start,
              lead( d.ti_date_end   ) OVER( ORDER BY d.ti_date_start, d.ti_date_end ) AS next_end,
              -- vorhergehende Belegungszeit
              lag ( d.ti_date_start ) OVER( ORDER BY d.ti_date_start, d.ti_date_end ) AS prev_start,
              lag ( d.ti_date_end   ) OVER( ORDER BY d.ti_date_start, d.ti_date_end ) AS prev_end,
              -- Typen der nachfolgenden bzw. vorhergehenden Belegungszeit
              lag ( d.ti_types )      OVER( ORDER BY d.ti_date_start, d.ti_date_end ) AS prev_types,
              lead( d.ti_types )      OVER( ORDER BY d.ti_date_start, d.ti_date_end ) AS next_types,
              -- Auslastung der nachfolgenden bzw. vorhergehenden Belegungszeit
              lag ( d.load )        OVER( ORDER BY d.ti_date_start, d.ti_date_end) AS prev_load,
              lead( d.load )        OVER( ORDER BY d.ti_date_start, d.ti_date_end) AS next_load
            FROM
              _data_over_time_with_border AS d
            WHERE
                  d.ti_date_start >= _timeframe_start
              AND d.ti_date_end <= _timeframe_end
              -- TODO AXS: Terminierung mit Partial Load zum funktionieren bekommen. Sobald Ausschaltzeiten in der Timeline sind funktioniert Terminierung mit Partial Load nicht mehr. Liegt vermutlich an folgendem Filter. Dieser muss weggelassen werden. Dann kann auch _data_over_time_with_border übersprungen werden. Aber Funktion scheduling.resource_timeline__timeslots__search ist anzupassen!
              AND (
                          -- Filtere Belegungszeiten aus, welche genug Freiraum für eine Abarbeitung des AG bieten ...
                          d.ti_types && array[ 'task', 'task.buffer' ]::scheduling.resource_timeline_blocktype[]
                      AND coalesce( d.load, 1 ) > ( 1 - _target_load )
                    OR
                          -- ... aber ignoriere dabei "Ausschalt"-Zeiten der Ressource
                          NOT d.ti_types && array[ 'task', 'task.buffer' ]::scheduling.resource_timeline_blocktype[]
                      AND coalesce( d.load, 1 ) > 0
                  )
        )

        -- Das Endergebnis.
        SELECT
          d.ti_ids,
          d.load,
          d.ti_date_start,
          d.ti_date_end,
          d.ti_types,
          d.next_start,
          d.next_end,
          d.prev_start,
          d.prev_end,
          d.prev_types,
          d.next_types,
          d.prev_load,
          d.next_load
        FROM prevnext d
        ORDER BY ti_date_start;

  END $$ LANGUAGE plpgsql;